#!/bin/bash

printf "\n\n\n\n**** RUNNING BUILD ********************\n\n"
APPDATE=`date "+%B %d, %Y"`

if [ -z $APP_NAME ] || [ -z $APP_VERSION ]; then
    source app.env
fi

if [ -z $BUILD_NUMBER ]; then
    BUILD_NUMBER="SNAPSHOT"
fi

if [ -z $JOB_NAME ]; then
    JOB_NAME=$APP_NAME
fi

if [ -z $STAGING_DIR ]; then
    printf "STAGING_DIR not specified, defaulting to dist/\n";
    STAGING_DIR="dist"
else
    STAGING_DIR=$(echo ${STAGING_DIR} | sed 's:/*$::')
    printf "Using staging directory '$STAGING_DIR'\n";
fi

if [ -d $STAGING_DIR ] && [ $STAGING_DIR != "." ]; then
    printf "Cleaning staging directory...\n"
    rm -r $STAGING_DIR
else
    printf "No staging directory to be cleaned at ${STAGING_DIR}\n"
fi

if [ -z $BUILD_DIR ]; then
    printf "Artifact directory 'BUILD_DIR' not specified, defaulting to build/\n";
    BUILD_DIR="build"
else
    BUILD_DIR=$(echo ${BUILD_DIR} | sed 's:/*$::')
    printf "Using build directory '$BUILD_DIR'\n";
fi

if [ -d $BUILD_DIR ] && [ $BUILD_DIR != "." ]; then
    printf "Cleaning build directory...\n"
    rm -r $BUILD_DIR
else
    printf "No build directory to be cleaned at ${BUILD_DIR}\n"
fi

if [ -z ${DTR_PREFIX// }${DTR_ORG// } ]; then
    printf "Both DTR_PREFIX and DTR_ORG not set, pointing Docker images to dev\n"
    DTR_PREFIX=dev
fi

if [ -f $JOB_NAME*.tar.gz ]; then
    printf "Cleaning old build artifacts from workspace root\n"
    rm $JOB_NAME*.tar.gz
fi

cp package.json package.json.bak || "FATAL: Failed on cp package.json package.json.bak"
./packageBuilder.js $APP_NAME $APP_VERSION

# Dependency Check
printf "\n\n**** Mandatory: Dependency Checks ********************\n"

# Determine where the script is located
SOURCE="${BASH_SOURCE[0]}"
while [ -h "$SOURCE" ]; do # resolve $SOURCE until the file is no longer a symlink
  DIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"
  SOURCE="$(readlink "$SOURCE")"
  # if $SOURCE was a relative symlink, we need to resolve it relative to the path where the symlink file was located
  [[ $SOURCE != /* ]] && SOURCE="$DIR/$SOURCE"
done
scriptPath="$( cd -P "$( dirname "$SOURCE" )" && pwd )"

# Set GEM_HOME location to local, project directory
WORKSPACE="$scriptPath"
export GEM_HOME=$WORKSPACE/.gems
export GEM_PATH=$GEM_HOME:$GEM_PATH
export PATH=$GEM_HOME/bin:$WORKSPACE/bin:$PATH

npm config set registry http://registry.npmjs.org/ || echo "FATAL: Failed on set npm registry"
npm config set strict-ssl false || echo "FATAL: Failed on set npm strict SSL mode"

#npm config set ca null
export GIT_SSL_NO_VERIFY=true
export NODE_TLS_REJECT_UNAUTHORIZED=0

npm install || { echo "FATAL: Failed on npm install";
    rm package.json
    mv package.json.bak package.json
    exit 1;
}

gem install bundler || { echo "FATAL: Failed on gem install bundler";
    # restore the changed files
    rm package.json
    mv package.json.bak package.json
    exit 1;
}

bundle update || { echo "FATAL: Failed on 'bundle update'";
    # restore the changed files
    rm package.json
    mv package.json.bak package.json
    exit 1;
}

bundle install || { echo "FATAL: Failed on 'bundle install'";
    # restore the changed files
    rm package.json
    mv package.json.bak package.json
    exit 1;
}

if [ "$BUILD_NUMBER" = "SNAPSHOT" ]; then
    grunt build:dev --baseUrl=".." || { echo "FATAL: Failed on 'grunt build:dev'";
        # restore the changed files
        rm package.json
        mv package.json.bak package.json
        exit 1;
    }
else
    grunt build --baseUrl=".." || { echo "FATAL: Failed on 'grunt build'";
        # restore the changed files
        rm package.json
        mv package.json.bak package.json
        exit 1;
    }
fi

# Unit Tests
printf "\n\n**** Mandatory: Testing ********************\n"
grunt test || { echo "FATAL: Failed on 'grunt karma'";
#     restore the changed files
    rm package.json
    mv package.json.bak package.json
    exit 1;
}

# Build Artifact Production
printf "\n\n**** Optional: Producing Build Artifacts ********************\n\n"

# remove temp package.json and replace it with our backup
rm package.json ||  echo "FATAL: Failed on removing package.json"
mv package.json.bak package.json ||  echo "FATAL: Failed on replacing package.json with the backup"

# copy docker files
mkdir -p $BUILD_DIR ||  echo "FATAL: Failed on making build directory: $BUILD_DIR"
mkdir -p $STAGING_DIR ||  echo "FATAL: Failed on making staging directory: $STAGING_DIR"

if [ $STAGING_DIR != "dist" ]; then
    # Need to move the grunt output from dist (where grunt puts it) to staging
    # This is a hack, ideally grunt just builds to the right directory
    cp -R dist/* $STAGING_DIR ||  echo "FATAL: Failed on copying from dist to staging dir $STAGING_DIR"
fi

echo "copying files from project root to staging dir $STAGING_DIR"
cp app.yml $STAGING_DIR ||  echo "FATAL: Failed on copying app.yml"
cp app.env $STAGING_DIR ||  echo "FATAL: Failed on copying app.env"
cp .dockerignore $STAGING_DIR ||  echo "FATAL: Failed on copying .dockerignore"
cp Dockerfile $STAGING_DIR ||  echo "FATAL: Failed on copying Dockerfile"
cp docker-compose.yml $STAGING_DIR ||  echo "FATAL: Failed on copying docker-compose.yml"
cp docker-compose-beta.yml $STAGING_DIR ||  echo "FATAL: Failed on copying docker-compose-beta.yml"
cp docker-compose-fortify.yml $STAGING_DIR ||  echo "FATAL: Failed on copying docker-compose-fortify.yml"
cp server-default.conf $STAGING_DIR ||  echo "FATAL: Failed on copying server-default.conf"
cp entrypoint.sh $STAGING_DIR ||  echo "FATAL: Failed on copying entrypoint.sh"
cp apply-app-config.sh $STAGING_DIR ||  echo "FATAL: Failed on copying apply-app-config.sh"
echo "Done copying to staging dir"
# Set DTR for Docker - Perform against ALL Dockerfiles in your project
pwd ||  echo "FATAL: Failed on pwd"
echo "DEBUG: staging dir set to: $STAGING_DIR"
/usr/bin/perl -i -pe "s|%%DTR_PREFIX%%|$DTR_PREFIX|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Failed on set DTR Prefix in the built Dockerfile"; exit 1; }
/usr/bin/perl -i -pe "s|%%DTR_ORG%%|$DTR_ORG|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Failed on  set DTR Ogranization in the built Dockerfile"; exit 1; }
/usr/bin/perl -i -pe "s|%%CONTEXT_ROOT%%|$CONTEXT_ROOT|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Failed on set Context Root in the built Dockerfile"; exit 1; }
/usr/bin/perl -i -pe "s|%%CONTEXT_ROOT_BETA%%|$CONTEXT_ROOT_BETA|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Failed on set Context Root in the built Dockerfile"; exit 1; }
/usr/bin/perl -i -pe "s|%%CONTEXT_VERSION%%|$CONTEXT_VERSION|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Failed on set Context Version in the built Dockerfile"; exit 1; }
/usr/bin/perl -i -pe "s|%%CONTEXT_ROOT%%|$CONTEXT_ROOT|g" $STAGING_DIR/entrypoint.sh || { echo "FATAL: Failed to set Context Root in the built entrypoint.sh"; exit 1; }
/usr/bin/perl -i -pe "s|%%CONTEXT_VERSION%%|$CONTEXT_VERSION|g" $STAGING_DIR/entrypoint.sh || { echo "FATAL: Failed to set Context Version in the built entrypoint.sh"; exit 1; }

rm "${BUILD_DIR}/_temp-resource-directory.json" ||  echo "FATAL: Failed on removing temp-resource-directory.json from ${BUILD_DIR}"

ARTIFACT="${BUILD_DIR}/${JOB_NAME}.BUILD-${BUILD_NUMBER}.tar.gz"
printf "Building application artifact ${ARTIFACT}...\n\n"
tar -C $STAGING_DIR -zcvf $ARTIFACT . || { echo "FATAL: Failed on 'Artifact tar''"; exit 1; }
cp $ARTIFACT . ||  echo "FATAL: Failed on copying $ARTIFACT to current dir"

printf "\n\n\n\n**** COMPLETED BUILD ********************\n\n"
